<?php
/******************************************************
 * Single-File Auto SSL Satellite Installer & Reporter
 *
 * Logic:
 * 1) If not installed → show "Install"
 * 2) If installed → run 'info'
 *    - If "vendor url not found" → show "Register" form
 *    - Else → show info only
 * 3) If user registers → run register, then show 'info'
 * 4) If info is successfully shown, display "View Report" button
 * 5) When report is requested (via GET param), show inline formatted
 *    data for each available certificate (from "manager report")
 *
 * Debug logs are captured and displayed if DEBUG is enabled.
 ******************************************************/

// ---------- Debug Directive and Helper ----------
define('DEBUG', isset($_GET['debug']));

if (DEBUG) {
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    ini_set('error_reporting', E_ALL);
    ini_set('log_errors', 1);
    ini_set('error_log', __DIR__ . '/ssl-manager.install.log');
} else {
    ini_set('display_errors', 0);
    ini_set('display_startup_errors', 0);
    ini_set('error_reporting', 0);
}

$debugLogs = [];
function debug_log($message) {
    global $debugLogs;
    $debugLogs[] = $message;
    error_log($message);
}

// ---------- Base64-encoded Assets ----------

$museo300 = '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';
$museo500 = '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';
$museo700 = 'd09GMgABAAAAAEbQAA8AAAAA6RwAAEZwAAEAAAAAAAAAAAAAAAAAAAAAAAAAAAAAP0ZGVE0cGjYbxzAckmoGYACGLhEICoK0UIHuCQuHFgABNgIkA44CBCAFig0HmR0bMME1yu5LKNAd4KRi+UUyCoGNg8D8w6kjEfaKs6om+//PSDrGkGENpmp6X0CHRo/BTApqUx8g03SeTMKyFblbpeoSPD+r8s+JSsxlac0kI0S3/A+QBBWiN1l3KLfO7c64/ypft3nt5UFa8ENiII10WBIRF81d+NENFh9e0T4Q/I6XS422j73Fozz8uGS5jS7HTc7AtpE/ycnL8/zv97R97ns+bhLSpEEjq5NMokmy6JYhEgkJ+h+ed1vPr+XeiICDIeLabAFxsQRERBBwMBUcC0e5W2rD1JatcbbW3VVqdVa2bFx3dXWjfcvm3XVUwW1u3zwxTjzsL52fT7TSCtAktLwG/sD6qANi3wXLdDR9iprJZ4fL9IQVkccH7ADp/3SWMwnPT1iSl+gAsA6BpNmsiiMqSsDOTjrJPiLLiCvzsgwoS3mpgOrw//QXO/f9Fcgwg3BnaiZgCDD+iS7K96u6Z0fo2A+iEbBkV9X5Ons8rPl1MeZ2SR0UO6DyyC0T9IBT/5e9xsmI1FpafVS2EO3UxgOeFDaweyn/rgMALP//bL+q5W1pQ0nqd2Mu8AX8wQ6Fcz4hRlbU+jKm7HX4N7hBLNjtEwoxUeryPwDAr3R+dWvsv1X9NOto94Aouoype1rSGA8IpVRO7QOEFAizY1W3Lwum+40DEAqU4wIhwagG18LT/9qr3fv/CbFQNAFe4RLrACcTopew3BPJLGScBaC2MqqkomSVqTCa/t6mVdptSQOL1qKOZ5Yhlz1jzwLNYniQ3UUXqt9Xt9T91W67e2TSIJsWPPKCuv/vL7Xk8bRH3lkCmvGQB8iHsMvRAfPSZciVA6aXXXjxJSlBBFm+cXzhxdH5fiv3Kns/3wJK48ba/T+ZgWSX81aXxN6JQxb/W7PyETtgRMy0foLUfWqOYdd3vpV9Nh3vlZLplb1C6gbXFcIEYYQwISxmyF/pMDb7PxbXu72qhUqUgYKggvb/i7oRbFJSRLQzvziuZPoqBP56cAYAfv7Z9OXPyApTsR1QNJJ6VmmoBZTW2ljFkgWU9q6wUqMgGC2HY/XorxDYqWGSCPLIH0NghyBqBcU4tLHqV7PBk156mY1VeHi11dvKRtrU7g53spnme9jz3rY4+EIc1hSMbkzTNGfXxRpXcbZ9u/foXt1H+/ag6tSad1QHVbefjWf/QdWnz/z59SxehyVBDesWXDFNzfWM5VzZ1V3THbr77wWqBsXu5YfT2zOe8qfqWf1se74ZfHu9cMrHqwsvJnhNvk9yV7cu1pxQe8cb78fe1J6N96x6DW/d2/J2igMhXZoey1DbZaIRdQOdICLZtZqe1pfCI8PC/mkSKy6bZM8f18nnRU0Eouodi25Xas86/jwgsiKyzbMjnAQ34KVEVTuWt13pPctF0/IZEZRs7sVEBGnqBVKThAZU4zAa9zWRHUaosGZnh9UqNmct2XHKUDacr1JpBfhRQZ4JPjfCOZB0ouwjdyl6TENNl4muJ40oGyCsLtWQ5TJpAx7t3v/EQ35QogBP69ZS4BhBj2yoakTYQB1MF6B6f/mzHeqq6anr8w3qMrDZReqg7HpvRZS/ZmH2p/s/Xzi+krbjL5Uik2hV8xOCKAKbCMRMB/J06DUthC8lhiZdlHZZw/W8YeUVzlbJ2q7CKzLTLefTzYICVM+CHC7BYsixmI5OMALktSzIVCkpEEmejObtqjsf1QDNLUYqzCAhr1qG82ySiGkJ70G4zO/ocw1IVEmpD6Y/zcWZ78lCRtS6xVzL5WLNFztBkeEFAY6Qia0UMllF70nIShnthHmBCLFIsARZlhezdOjDCBdvWDyom4XK1JIReQzWCexJwcPsZ9xbCDLfe5xVitOSLpewMhcp5WvXN9ogcdCtva3AUCnTGMZVlAHD0TIpufmaRcoLhhZxCwrNhMa23kozfa2wat1J7bWGTymzCReJuqLNqwh6BMLGBlPGjHAPtlH2FELbZaFlVFfVNANGjDj97x/L4HEUIoUsjRDFHB25YU0IxyQyEWiDqe2n7R4NR5ZCZMVcHfYwgnIiQLBsnJgoJltGbBTbZBPFDoqdFLsodlPskdaxQddJPYf0HfUuxa8UA7sMPeCi3S477SrFTYqRvf6g+Fvav9L+lx5AetihhYZ329QOmcIidDFdh7pBGWXIgHHaRG2yBqIl1tbRGrZjHLkdksMiTLG0DmtYGo8jEolkmY3j47oGJgI2LNtug55xppn/GrO6tuvZqG+zgSVDd7toxWU7jawN2ziOcIeEWDIeimqQuYuyeG9IKwgBDdM04z6mp6WvbMA1tMNFrvFTRqquZJyZanddtulbCtjOQX3cn/1si14q5BXcw6uFdfl6An2mt7D/20W9U8R7hX1UBZ8U8Bn0i1y+0q+R/m1B36f/WMDPNmhld11ykeAyx/WoEX9YZgprn8Ywzrff3//2wneT1K2t/OtD25+5hcM2UFEb7MPuLYEfCDBwIts400M9GNs/5iLksrz5GUpc2ntkGAyb4+CtTzKa/RS+R9T/46REY7CkpGVk5dgcLk9ZpWYioeFFI4gTL1GSFKnSpMuQiYqOgYklTz4OHj6BIkIiYsUkSkjVs+jWo1effgMmbLLZAQcd8pUZl10154abbrljwX0P/eoFDnHRZSO/+3+mNJvDEwgRJWUVM+YsiCxZsWbdKMtd4Xhr2MRP/czPRt9hgfnSVTCrBZIx1QIN0sYnvkTEg5EI//HTA0lRjtsIxMOEfhqrYYmey48Jkn10y2tTQ9O2B1K/BPeDcZCU5JwN6WjHRnjiVf2eJhOyAldksZGVCeO4KBE9IS0y50tsjNcxWIU31qQFJk9AbCSreOEvYEARecMgcIb1j4uktFNkxqBSn7KJx/BOoLbYAx/qRSA267A3iPEfIwtka8rInJRFSNn0KmavQCNpHRl0G0gp7QKohngJjTlIjEXxCiPj6ERGCMy0ylhcV9IRQnzCqJWcUOEEzMya4VnNqaINmfzmQVIJAWAUe12IhQX6bkCoAHIMSauoF4YAso7lWKLnscRnsFC+xP1lbr1FQ2jjm2YJBVFVNx0N2KMV2h4JJKiBquvmQTIS0ppZTsbbYdEa+J6Kkprm+GncLpdoBwdB2EFw+kffvGYQLcWbFfdWqH9a0Mu2+uqjdwYAidrb5iIbobdaz1MGSiS2HfCiYKqPk3QRecRdIjZtfQtUZUO8HKx5CjrbKh2fK5VQYPO5+R2It4mBbYjiFYqoiBQkRNLWsttruMjs5I9rWahsIVPxgxUxHvjMVmvQlJWyslaC6oCRRY/p8QU58TpLEChjMAHHilUGSOpyWAMzPQsH24yq/XtiPi1DxDfTUL70nulJ97MaIul191tqMYxQ6nSYNZZUia9qEmugHUXZ81fXeIgnyp5k/cpp+zGVWYEF9mq4SDH2NaMH2ab4zWF/hmAj/ce/Vj7IliHmJ6ry8iM2OgRtpx3uxD9G/OjgwCQCtL60VEA8kpQUsf8oxn584hs04tE+3aA/8vGuGURF7fPJWKptK82xLA2Ng7x3lcHkSvJRe9TT8G+QC0oyoajpl9qF2CSJWkcfSbKq3sOaSZ21yYwE9R1U6vi6hitYfdDqfySsmX4cliLthSh/yWkM1h2Jg/AENyo+/DKZQqXRyQxmxy3TdxsH8kK+4ClNIIpK/WHL8gAqwNUu3aXb0NRCjnbJqddNMR72yVgoXcKmAIYsdjuFmQGeMkAIwEvAEDzgb8hvyAkG5oXrCqBfuDhpfVmpePApCRXpKGcvArJEiGGYoYztYwAplYHTIDjTP9kt22pcjQ09TQ2yCeP8JwAwGJMApnyaMaUAoHhPhdZNx6ArfEQKt1dLAV48NdB2a5w8xEJWgPpKCg2j0ZMxF9hmfDH1BgXakE44wkcQPFdWZCOuElyTb15f1lDkC0jysmQmhQXss0MoT73MB4TRPfGmABj31AU0AK9n2QcnVuUbRc0jrN9hFG8YGCuvTysWXx4YKx14GqjMdFE+WhM9fRQIZaSunlAV+A150gFy8uLNH1nBhgZrmAuyLUeMDATBk+uZ4520OM39HJ4R3V6wyTbrS1vGYdgRMHXrpSmjZBW6mcIXQtxfsmPH5VxUHq5w/B3jiEhFqZ9TnHbrxNtgR4a8CzsWmXKc2Mkdy5xzXZl5f61bfpM7frMFf537/noP/Q0e+Rv96G/yxN/s6Y5b/Oy5Vi/9nV77u7z1L/OHf7m//N0Wu+rxPg+9eSXAjoKqgP3BAg4EDzgYIuBQqICvCg+YKjLgcJiAI+ECjoYPOBYh4Hixs3Gi+GQ6GakKuBgl4FK5XZouv0IzceO5kqAK+K6GgLmaAq7VEnC9/qO40bo2mD+rs/+7zRdMEfdx8x4N54yuF6gv0jGw/vPKRUDBjc8z6G6yMZjGkQaa3m96++ex8+5vSw8Y/J4fhqHfGbCBFHP83JzTGMLHALFx8PBM75njmeGZ5ZnvmffqMpu/eOmo2HgkymmYtFhm0Ihx2+131FmXzLnjkZ+9tsgK8TsbJB6WBl+ez+Rb6Tvq/XXNgvLpvJfWrMK9I2vWLtJ83ZOsYj9izSbZZ/Lt9T5Zs6V6d6/ZOfkMb47rQDbz2ECD9bemFa1sVavb2FAbGm60da1vrJHWsjKbo6lmQspEJCWjUK1eM0v5FSRKUgkrx1nptaabDwYFg7AW1i7Jr/nTotlZQK7VBHKjRlDNN0y4xEoMby0vVvxcfm2CBVH/bND55wgCXhIVw2ArS9NSMSUS6eqVnwuyXGTgy4kd7+tEef58cVfCiB/IWeIZgBqcRE2fR0/w3rXYEmvhJSNjB/tkHw1LLbUeQSqqPMLAlvijezlbG8VKR1dArDywZ0msZFeTRwKZGDgkVAwDPeOQ1ezr+UiKhUeqQo2WgZ592FoOWnikKyBTxaRN70AuRDQZnr04IQt5a0qk8b4g33eFysStqUpd/GEB+6rxJtrU5ra09bzB7dr2djTV4Y50tGPtbFe729Pe9rW/Ax3sEEir00HX3MoDKo4h9gEuMRyJa9rr40ZO/HgoxF6wjjrralnL666n3vrYekBJFTFDgeuOd+wkt/PKJ5aMxABeGbDBhzhkEplothQrAo3K6+3w99fui9zxTnSyU53uTGc71/m+Xo4GqRiQP6xDxjvEi+f+2Dd924UudqnpZprtclc8Pe6PlWpUozZZZclTVJ4S9Lg/pkpdRZVVVZ0mbbr0oMPVs5xT8pJKDmFShdmKvUC0OWCKpebuCC2hfD39gOemBeQfsT4Dxbjrt4FPnwa47E586o4uAKz8F3r7nd/P/mzgwKsA+X/OAlZyRHUPzFtG1rLmvXbbxA5wHJyN5dpM2pGloQkss/PDnv154ozhc0MnDEISVsKJcDHcCDfAG+F95+6FEJMEZCtChlD+g/8vWGmLHXY1NH5lZu7nbn1OmAu9MPREA5vgytoLy/8B32lMfpaUEPfa1f+WP705tezbZFlkwbwZTUXOr87P8+08Pb+u06FapTN3H4p/FDqjmVNI1qfQJaE9FPGoKvatKC9irwoX+lNwcVZ6Umr7FFli0lS/AM+eJXMuhGWvBYWIhIyCioYu+2zZpBxsuSfMrp4ChTi458ye31IyZYCnLdejzxrrjNtqi222m6zpdx977LbXPgfsd9AhU4447KjjjjntlDPOVsSgnJI6kTqb6VRmZFKRWbcN1TJWVz0Vs/H2e6gJ8KGTLBQ0QE5mAjyiOTG5dl3W5hHP4+d/t3nxK4zjBOC7kiPNkKSSGmtKqD5ugop16tdhQK8VVlpl0LARwCs2GjPqz2jl0JcdI6aG2LGiAzK5KJA/AarXgOoysPMFjn8BcBRAA1gtY+QaPWbOxI+2qeFv/Kg7I2WK1/5MXDPTSbCXqXkKl+Db7IcCebRWeCwkilf+mLYRmczr2Yo6y+VXnNGAX5IHR7qaE8VB3B4JEfEi0ApZztIqIUdPVpI4lsWUQp6kq29NMfMmycqQIicSQ2rttI8zqZGoUE+RYOZq3UnPGauJj5xDT8MoveK6jndoLG6Pr4KCXiIkt67keklDMzzfIp8P1m/66XCUObwSH3Ek25HhzmCqhrlv1iIpjOMcPnIajykVgi3w0N5Oe1vbfI8ZgPfDSw5xb1uba1nGNeXyD782wGF/TlN96XYQGiO0qzIRUYbLGacIbAYzUOVDFH2GGyJHHdniC0fOZfCGFkf8NrwwpQ2/tIB1xSV84Qh/G8IWthNeUeoKpORRnx7CF46cQ8rrWfzV6JvE2HFMZdf0bNlDBOAIf+F9QiIXSMGVI0OPWSZg6WfrCyztD91xwp0n/uyMgXuTdXczLuXmAvvv8fvn1sjHs/IeR84LvaIQVXMLxsyEE9x5wk5P6Yj7iwYznb36pAYDGg16gr+z73NBRlFWYWYTdb5KGSiUHBUKZKL5pru2mgNmHg2zoDvnzI+o+MjK/8ffo4dh7iwaL5MvPmHBvtrigooJxxa4P45cmVJYifKZ4ydouXxUFVVEOiwmB15wqLEkB4b4CQ9S04CMvG8tzmO3GjlczpfN6kKEcUwspD8cdUUe3lEHuXt61t6WEvnv15HNJx6fkkfAeg3nl9je9tansd/Hy+jX9UTPF8H3IK6Wu1WHYRw0XtYWLvA0ickbVMqgmeXlsXDx1aGrq/QSV/LdOfpamZ9fBrcwzW7VMC3KoI/769ap028S1ZF+1Sli8CT8ibn5P0HIjlP9szfjhj+WJa+waM9H/2l3De7fMuy0LUSpSvqS73H6/XEc81qXYllVidpIB4pBIKHWIDJp2wdbOKQ3TjDIfmFS/IQIJr9iivgZkZJzcktQNyXTJM7kE0jYnKrlSDeNDrodbvmzz8bw/333u6RbhvGnA9Mes+iwi9zb+J3oGIDwCKaF8tYiOu8SZb1ehkmuh4utLESRFHiL8sD7yH3tS9nSaRIkP5efIuBI+vPP6T5SVeMovhiFFEkJ7PLQRr5IeV+YpHMtMUySIpW6IzKMpVP0lEMSAL/5sKxqRAaPF7mrIma1gaAb34MBm41l0jnSJd5ZpQ0fCErlWbU80z3QVJMzmiLnMdf6fLp7n6KweauuCz1ds97nRS2V51BdmLnMlMQkSQyqGFSgcjMnnM8POLuXOdKcI53K5/N5vCDnziopjmsqa4+S1I3q9TJKicLMRwcqeVXWI3mJtg/8lw6EXkt/f4rKDJPL5KcRHhKGDBIH0XKmOn2wXLxdJCiw7OXCQGbYVJ9nWXVqrFBmk3M5m81kub6WL0Qhemctm5W2UAwjlQQSinRyFHSC/MvasoUJoxDeQWQ5m6U2bREDBhLKgimDGrKgtuPo4RSORRvfEhkV1dLIU7ciYYba8Oa4BBNpkvy0dfZHzeMjs6GkC++PDQ7CxAyTLnogPjQkfZQa5ZcsKz/XCJTu3nYDatVYAgpZKcvVPsw0Wc/kb5Pbh2UIedcP6VRGfUbrQWnUoDpSfQc27k2bPlI1e67wpvCIZOABRWrgJ0R3/f6L+v7ng4MYfvDlsASX3OwlP2ZZPB2XYJxB7iGdTHNes829T2Nd5AZmeHUHhaqWJCa8tJWHcC5f3+8qFOWWkrg/qgBfbOT8q+6P6IL7vAxDHhhkm0AyAC1uw617pef2wYZ4Ri1NhRFtniWYd4lWzVqF4o7ImJlJrlwBzWWEdxHHPW92rZDUgptZECygHeFyBXKr9TF5RfNznCioopSswUmLBhuRQzdsMa1Acc2XfT6WhRMHgNOasmxNYxS65XPsoA2d4YkuURtSGNZUr5aBk5wSI+M07GtZjR3EWpGYHvNKUW8TROJT+BDOkBxJaYBGQrdAJEbuoY5g5Dig+PkMKAatkV9bIhcrYE8ugrXW79BHJSFfcOuDg14RpasmJLMMgSUf5ydS/wAKSajyazsBai2bU0NIaILSe9YK/Nz94pv2g4OFukfepTBXLlGzkfM4eecj1BHNNZfTmqYNXRmvcllNB3WbKRZuhyY24CA5w0bRNHRcKXYM05GDqRPgQb3+foMxczalupy6Gx4OSz9eLpU0u8u9gdkzsmJ4khNsPzy0UdBakvnzYHpIYw3wqI9/yvJkDnC+t1HnoVcrU691QTnW+Yb9Ksexw+UJRLBWFGyNXgVRbbCteodIUXy+EKoNud3p3oKwCRt7rYUbeiTAFjpfOew9P2FVJSu7wCm90Ee1lXXryC6/8gZRxtAnSXa/Xa2H5scdJnC1RNWwtpQoegwaZ9QL/sX0Dcb93N/Fu8pqaq81ABFxIkqxhRcdrVxSAj3nBnZ/jB1EYg+FSuFDz35hGXe4zz1SQrjofpqO/sT72q27q8ljc1DfX1VbWn8Vg0r917tv8p0z/k0P4KD8iq/oljFVnc4HapARc5egBK1xstR3MZ3D1pIMc/i9Hi5o4mvFZalJWBzi4znSxQ41kfSjfuZYCLsgizRmgwE3Ks8gZVCwnneMsQzc9hPczpH+72I1u59k8luiVdxf5geW2mqVbmtEWYY5x+UKBGoUF19fyw8Ayr05NOSKooPfxuJg3xKCFHysnEihc6iv8R0kv89GmuPkOWoRevO2BYNk0HV15sfTrm8XkDqRkXXiBmOSNWNgNmE2nkNR8I14CVjsPMIqBfU0Pd8TTMvvFxLlclyDN/lCFs5BkaDOeg+5N4AAGb94pVYugp+AIMeE0vScVgiYrHJBoY5cxAH9NvJlbSioIf/Hyu+INjqrXR+AQ6GMre817cCFebDVyEWRDcWq2WwtScr7Judaz0EDqR93BDD9di/iI9/IW7nFyiDVTMFBT569I1V5Ajd/OZ/xsgS7HNihYs5Mt+ptqQatUaak8BNIEECo6K+EMjozQh0ZimD+CrChCniyT9VGDIHAmZKjDtiholzkaPvC1AuZGe+e3HpOjaQlMFk49vSjvm9BkkU9w3+5uhMv6KT7/HlLuPgLa08JQhPCnQN0KEhMuKteSQcxYBJLIk4IEXkGPy+OIThMNBu81Wh0eKM5jvWOsWWzi5koRTAb3Lbc4Amqgik8ZJchEOsaUxxCXWSJLdZVtnuKukxgwAcMAXRO5JR9j3CSKCtvfjiHPEPtLGNaT5Sq5tvyOqHhtpd8kByqilVkj42oUqgP/BgsgB/LsJIjUWfJEBX2rgWVHBc1+RfJf7ZCNGR1t40wkO/SeKAE10L0SaEQyFECJwIps+Ro+8Ik/YKfU+c8ovlweeTnx55+VO4aEH1x1BnHyppjVG0mnOqO8+6b5MAUVEm+C4VLBZD05BjJ1dCErEARhmN4R+Ml+e/oQIPZmqud4CzlaGkAb92VYrOtV+bdG+rymVqvNbbEvPYrEljOw66m/WFMylNhdKwtUSlE2Z7KwFdqH30vFNyqbTpihdulhWDsp+wgMeiUM4gO+BZ2MyrCWE/xqH6XSZ+82QXcj+wkCY/OYym7PFA3U3Bi6pkVPYug6dhFcppC3ISefWp69typJbKV+JJhYMqQjHZdZ1qLvPfotzfK8TAlGmFkHWrabTukUmM0HidpMcvxETGUYMUl+5WiZcmP0Jiw4iw8mPISHoyoa6eMnymPG5AHMPdA9Na0qFQiNKNQ2jQuPOWiRqsCvKkhcjmzM+JJl2EPg9CSE26UL5RAvZGaSDOrji8J9Sr7iZOEZuG49R85NMn2lbmPnTKo1wNJR1rwMpNL4Cwq+TRpY9KNNIEsFV72DBlRm3mKWTu77CzG8DKOjtowQUjW8IDPapQPOvNwjDbHs4rXhLthqP6aFncC5hCs9gq9ruFTgzNIh7chz3dmKgyHaYaLkFaaUA/uU+GkpJ8G01SYNEJFN0WDbSeuNVPWqHZFQ2njImDv2zbVDDTMhmKA3BGOnmb+SMDa+ImEhPaFyAZGzOTSkomlaDR+lgGdQ2XZQ43M2GwhxBXuRoZF/bjCiC+0GFXG734m9WwHCCVNEEOPmRpHm5QSCVPTGHqhoh4QBYv3GS7EgFEzmthu8xOsoXlhyB6Q2YjIyQ9QL1Ha1DROHD1pW67Po6h4VkBElpbCckoaiHMJudf3ySUakg5QzUI4ToSmmNEh+XILCc4AiMRVSB3z4BhXsu4iyYM+AgyBSUc4kuXhiEgInSh2BBgCE/LICg0tMhqBndgkkSlIMSMW3bec58SYpy/nNxoeggmb3O8DajYWSKXxBLpFJan7J48AitFUg3VaPGnUlmrhFXwFqkUqiKYiAZZ0gFqUaBF0ZGKS9YxpvWQe7V5vqwrql6v7o4U3fVKfV5BUj5lTcYlMN5HiLTA6paBLJ01vqKw8ZS9VkuRuCLUCRk9oTNsQuiGy25vXPoI1q79JGwjLQaN1X1oneTmYK712K86+2QTmeiaO84i+wggDANV++kpeepgWMpejcFBThL031wmpH+ocfKVaU9Tj/7I/f2lnthZpJ98MmrijRSheknOPCUZEV3sWQqS3mDTXD2YAS0e4OB84t1MoSHbcdsuz4FN4a05I8ctculd7RDrAhDmq8jEVKR+y5DSlmnX0nSzEv/t6ZVN55+yiqqzl8mi859A5sb3vc69b6Ip7T0zNI+tM5jwF7o0TLwFCU3CQN7OgkJEswuzZdNnCYUksh1AwJRVKLgQJpXlkien1UsYD4wRDinkZO4Ev7xsFXl73KBUbrem2eHGG7YSHwxRqSTsIL0m20/eCXFi0e/DfdvOsS4lr6FnxowlY3LW6DDcswi3dT3cpgq0sdYNUdC7ipT/LF+x6DC01YVsHYZ0HDZIRiwEmxdoQqx4VA05KTEaDUUaS0Q0CV07gFJzPN1+QFUrzpMm855hLnHIkW5xjzjnjiNcYnIKxVARdJT/pBOTbcvgrpvmn+a194yItZHtl5CCtrxp5+4bwIV3rT2bAl2tPpj9vFcRnmnlcfjdtPjA1/6Mzj+jrKQPN1gCoJ1a3aF5gqG8r/oZEbAV8qJepRPmPWs3wouOdAP7WsmquOdzYuDG94qUazhXz9CwU7aQlCRzuoQzBvIrfom7TStPHfkW3VTanD3kglnoljXB7ezMR8jknfSz3qwjSRoXm9XUg2hQUBgUXles5o5Lb4keguAbM6aX09a8lu+5ht9z6sB/K6VxtbLFqNrdeK2+7CgRComhDtWIjoSpvRDR5yl5E+JdmNeKdZjbXYnEZp3VYYb9fVgQfleUYzzl41avQ1Z4Ypu2J2zpBVsigFg8FWYIPZg5a0rvl6cfCp/uVzw/FwADFryw8BEFpeoml2XktW3J28eyn/j1feLPYsfDLqv9CujAYw1sloJlz+LjM/qNprq65Dtyzuim9+cV2DZpDTte7iOv9BswuxXWIWZXAvS8NGl2LG/wGrus3hJxWwU/34a0b793nN9AHcDyaDHGKqdgchD2LDqXFyiPlMbSi9Fy1lussFUjiilAhIUkVddsLQyPxN8LgVIIiQk6gijLz1Uaho0Qg6VbgucBpba+0mEwqlXTLJCRy1J/waTN8eumfycqgQFWyIDAIMdsplVAppZJOaXGv1UQQ//U/7AB3OY0kl0OE48/bIhHiVpnc6+6tCzCl2Tf6ZlS4b6pRYpFIDBKerqezvmagoUuBo6QmZpBJxNyGDGo8SLK6KbsJTkjrLvx9q+5MteHczV8MN0CrtNpw6+lNw7nqujO3/gZbiGISRwG1wkKrGwh2nbV2nQ0EZJUlBOLKxWSqFgITrxT336jf3Ffcf61+DdwqyC9aS1+A3F/0UXpAslgsYI/rMy+t699ewOI4qEpKVgdYCnRW2cqkFGWAZUuAJUWNzvj8+cPnDyBRv7HfFTKolufkT+01Fm3Ug+QS93iUKIVIT6JHT1CDou7ifb+6LGRkl2aFDbXbcGEE+TJDWQK+mJ7OSaPjPruw9QDBIcnrd3EjMEmMNHwiP4kcE0TqGA2DyJg7+sOFBy7nOxg1RbKe3hpMvidrYBQEHs+P+Xp0u6RacsJy4mLVpUc2TOgdZvIpwHBJMo/e5xAsIDGfnJhUQMkM0i3M0v3harN+qYOJCLlia3qWKypX9+sxuZ6B1X2A7miaODaW4qdPULFyNXhiJMPl/Ycw3XRSXqnGpQAKywkMV7X2KZQdvWozIYUSn0JlZhAo2FAKtwrk2XsP5/MF0E1ktP65OLEaxF7DR/PNOEYSt8Bs1CnbK4v5eSJxSomgp6XGONhiKvb499pBFvwYKZEUk02MjaMk39Z6wDwdqpxJ8phFCno1IToDPHaMzHZ5bxumm0nMk4m/4QhVS7+ivKOvQtKVc6omUHBwCq8qiZBKiUuhskBBm6VXWl86AbCKlBet/Bdhv/3T3G+R8+SuXrwSMtPqQSFH/lz+2xyLA9htKHB7ccCyOUJkKPlfO/cgpAXtz3vwkCaony2LSBC1q/h4JDMaExcc0VTv7pnY38fv39+8qqtR9plZ7F1m1rJzTNVy/lI+gVQ9vFVrHNnYpPjq+5vKyxwhNvdSDg58SDv2aQgs/zTds8M0FuRKu5d57vQbGRDPfJE3yASxUflmyJJakqIcqKi0TLainxY8xjU/7ydU4KMqCP0gUjk2Pz9OEHnumxsdzSjtsVOiY7Xtg+rKrpFqBMXjE3NDH0US/7bonNQvbuu120A889fOi2fYF8/tNOiqdK1tdaa2Nm3V1Rz26CNP/67//etk/eEVY4Pe98or/asQgFaMzd+aiBF57LmyezSrtOcXeVhsbXt/pbFnUEUQomJJXK3+pzodwhlCnKq6SHMNW3N1DohnOqJoUR2uFkPDgG7PAN+b6PbQVH4BMrHrhD55UdtpKBZ3GKMXFyrZCR+8XAjdMa45kCBJvO6qKE4PqCnT/f1SpBqK+jStYn/ydHvwINBCXlS7Gg+j0SNcB4XgX0Jil06jUzOdHdBSCmZj+0y70wZvtY7OZqMgUj16az556DM6cWsheQvLxrU9qy/10Vb0DFwSyp47j/JWy/XWzvI175SNyZmk5KRMYmIziFSN3rw5PiFeThubclX52czCvNxiWuvYrYVlY5ru1XJaeR6LWcBm61iV3QPl8m4qD9oviiPCDAWNRKxSQqP319itHrRrsW80xZ9OSFrHkAOkO90BghpfFiGPpl6fsMesyqwM43BcaSgQiCkLK/vjLMwSttVX4zieLWOj4E3RCz1X/6II5H+Z7Zil8/JW8/OyZzpm6yeG2zsmhgEETQ+MXS+8SCBABMLFwvHr/dA08J9ufUni29lDiB7Ty+XHp9dqJt0lkPScZnLt1PRy/UtHFmRvxydZXgL3R+rbJcOyAzgBlF4pPzByt+QH9RrteSwXSm3QnAdv0a/Gh3TzjC6jsSQVcodiq7XzQ6/G0SCcm3pxarNKBcWEYvRWnNRL/yw9Uw8kUuq0RDQokgAPtOvfzaSy7RydpuoB4uLl+YKhWtfJEv8MAK8YeQx4v6ju6crpAQFDEf90zbIcPALdbcVyOM0Pi40Wd9uoEYlKXWVZlhnjCVkNNfrXwBnlfEteYF0YgcyLcfLZC+gzlu8DII/KPVApTN00XN/QOaSG5S9Znl0EwbyCoX+IZEl6EoPHZNCKUhKZBQD+xbBhY0taWRgiNknAL6TCUDS+uLtIkh3OCgmBE0SIfc3TDPfwlg0bAOJLQ/uQKkTylJtAh5zcnlhmGjBukGfVAZ/OUHXTsKWOnVrA/g0QhH6pWz1SU13XWwmXlQro6ShmYFQwQYw41XuY6opqXrdx+cHHVj3YmA9AtiZLmeVLCKYfn33vv02z1VXO9dIqtWtAI/WPXGJ+GhEWAEVRZVVNI1pd/aqqEAmkoNGtl9hF1D/OiirNTZQExMs52BuuGYlWMNyWksnWOoWvBu79i8iQpGVOXWC04eNQ+Bhm6tumiJeAxjO2GVt5Xyxfzt0RWJxjQhgz95b6y6s7nN0X4Ixg3FeElqymIyu2PIOO+g51/0h+DDI3RbR5Ra4IhOsR3iGfYYE0QP/SlOYBMYVZyeaQCqhYmp6y6wLDQoi9iI7OIh4uwjx594o5GiASLwpExLEMMZGWmi3EZie/7UdPE9L/fOu619vWuRe5rh7uQC8SErOKhHRakahNKAK02RlTmAfkUX3WeyuiTN91qqCoxWV0OuQesGB98UGspk/PWDbq1k0oyCu4YcbyKBDyqNz7d2nIAaS6V4WekJ+AmVLW+q17FDvBZmvMVXAwRciXM2BhQWofLzgjkIGKJeFp2gJ+HD/IxRlfQs0k+iIDBb6eePpvBWQsGU+uFIiS5SgOKD69QFbShluGTUrTTNLz8DL9U7+A9/dmYAyQeYxhwcehcGS2gvHS8qUp1RNiSTMzpsCpk/91V6ald2ODoCBsd3plWvfx/8Desf+6hXlrIVZVSa6xJHcHlFolzFu28b8VfMQMlNbER4ClByF9lXB4xS6ryX9W5/W7Qp0tef2rJ//ZZbVCOAxAFC09GpweSMItvFtVFeZBher7FREDz1j+ZBkpIzB/FiCpP8PGBTwa9BP8nuYw7Z9XCD4Pehr6XAMs9LIg/9qB9na99dR35f7BAmwAkUxRej+RGhw8Pn5EaoNcmoxdNRB88gITFtwQaHNd6ewm/dbDW0khZzUD4i3nNOfABd28kN8Q7NJnrOnETo8eEcGCtIHWX/u+aYQVRG2nUDhsovAPKsN1/RNImjm+EVH9dMwn5Kew8uOjG//Tc/EkDIi765rq2vIcPujtPwAAAqmraeyhaXz6p5on4JEcFXyLEYsUFqRmSxg5NFF6AVJoxA7Kiim2AkBKLMI2CttrFAQhkkRiFL1MxtVEmKeKpi4n69/pUy+DPCFbPgoflpw9wcgnQk4jkrPDo3A5u689dBt725muM5PsydDVoLePUSTzK9QoZdEcGCw6uWAhC7+C4C6j5k/3J/B5+BNBUUOx7j1kcjm9SO/Jq6xQRHOCo4EH0t5UZvI3yU0g6iph/u7Iupt3CTcX1o3U//rHxh3G2vEdf4xvrzVS9dsqjYWItVErWawcpFXSM9icG66UEJMXFTDY2paD8NRNOu5EFILdbdjFKN0pe9v4VI8VO45/pqSTvVAkKjXjCg8HhX39agktMC3I1lES5crRnabCqADyn7zWOHltcq7xyeQc6ATgOfLn4apLgBjO0qfpDnW8qduQSMqKj+gwQ3Ht1C5pRjeh52K0/9n4M34EYEtxLVpaBOzX90rFZFIwP4xE/trHt/Cjv9ddIf4r/xdE70xzJmBY3x3djccFetX1jy14VO1dhAdlCRlZkbruVZOec1xwcjyKCX/yJIYc/7t/EGw48Qc4Mj1PVA/Funbw/DLkers0UnxGcRFYuZjT01XRA9izM2229QqfGWLtqbud82uYbwq/HJIh2oBLDLKsr/dSX09ZjP5uXyanhAPYP7c1t4FbfuO2W6zNCPwDvOuzMgM92wcXFEggAfJOruRGIJS853WzQcMrL+aCZxamspTJUEqZLKsw/j5H+SiriHArFNIqPMKqZZpKpgI7toI18fuNAoaS/GlJy3Fq3b/kopgjnvXUQKmcVMXs0WpdOHT6dmEtUCEWa+k80CXNGQ0j3aM9fOblq7Q9HetXNJj5S6aZzoIPuQpd/PWeuqa6yIkoJDM6LS4Y2YTze/mPnIGjvQfRi+rvlYXHizqUYauR69uaU8UZXN8KowKzO+lb3GOR9FTGFEADFcS8bvn1GYK/yXKLyBT86lB4JlzcGWt9VLHfCgc5RjSiwhojwhvDUI3FrghXNwQM4ebqAWzCp9F69Ok/9H98naw/SdhTH/1KwDrXfh5gPO6jxf3o3SLy/0mmWWsZZVfoxV7pMnXtVXUIf0y3qpthm4DbvFgwFsTa340M+Nuqnhr8VeuoAHB+jw03VVYUrKmxuMpUwtDgdH9FtZ97OvvfLqtcFmrn8XxXGTb8b+MUSj47CMZhlOR2gDsXlSU7aLM0PiyVld7JypkNnnrp/qV7VNqFu33INZq+N32fSv2rAyz2FzvjZfyofd37psiruBO3FsareLF4YjeB3KAV7qmg0IL6ZUR5z+rbd+YzsgN6sOnyYN54o1EJTYLxq0YVNMDdoVEBTXKc2Z4TOs8+RVsS7bVrBkhcx2pdf33nxvA4qRnQ3cW9KmlRvYIVcgx2Mhx7IjYbm6vmstFkp7szmhRiSEuID/5YdFjAYbezPkEnIjCb8Xa+/XGYwe0bGAncjKSH1I3nFmd6GOTJ5mlcWtjEN6hLLz3EHdFku7slBAyFP+ReZjgogt1txQpqLzJe1P2zCplQj2dB6SxtZd0KjU/JEgqHmoFiBAbjYkT9mrtgiKKHciwnmAE4Zjn/zOIjsBytzHcOMofHkHixjqgJEDrg2uNYRO9ot6syKl5G9giVtcTXIq9ctXPLUL/KIM0ZCyNoI2VOPxgBN2bOHBjhVI4/KTWDly6fbk7vixm56byx9h+ZPrb7Pp12/KsCAt2rIPaHgPDX0rblmiD8zuPQDSL6ZP4Slo5gS3JFKFoDRYnDmUuYReG99RcxbgURObuFoUmGrmFLWUpGYQqRWEqM+S8xk5LbH8uTkWGg9AM0Q/UxDzYgDNnHvNkAbuuP+bUBNyuDP6Ww8oKTyuf45aimRpN+WWGvyJhfR4CQL6bkJMaP9UqVOKf59PmN9KBXpZTltLcTi498/p+sCUV4apr91IfvQrKKhIM44xxDBXRJxnFG9umAu5CRczqEuL8kDzaHd4O+ec+tm7TZlZaH/lBCtkzduu55+zpVSMbfpsTB80fnXetpFo2XlkQXDVKFySmMXIqnxIPiGEbpm2MGA2lxSbqXlpDKR7tf5JnFp6Q7iZ2Cg/198cGOYqd0PqXxy5XPyMx3ZdpCXIxT38JKlkDjf233SyYmQN+zhBttVhJbIsOYvuCLlUhMgnxSwXT81ojCVSqrCSzj6iFkplzcl3HmyIqXW3mIhCnHSwtP02v0nPS5y6gsOQxB4Yplm2WVkkJ0f3uIhGYc3Z01kcx26vC97VBrZXUAdq00HgEPxN/ZYDy7PceSw2Azoik66/mdGPNzNHtzn8Hwvj18Z9N+6W6/wpqI8z34w4oYUctnvIxrpl7jbu3y9TJipqf9Amsi/iS2hbh0TE3BSpb8NQ6N+KWQ4yFGk5eQ+NSM4xWIehnxa4IQzx+3BvJwZ2ym6vZfLGlGqXMT5nDYL6+4Z+PsK32TxW7pLBP26ciniK689KUnX/pFZuzTpUexzSBt6dCXPiFmLtYV9sGO0yb8naw9W1nljx0uCctcVHMsGcRTTvk7H8J8QdZbZ3AzDB0GqqHTkMnNtO7EUDGj1CoqsMazyTmWnOXs5aU0DEv0SEGuxQF+jmiPUYdlBKkAbTghm5YKSOAqLwgicmc2F/mWJ4PK1mQMZ153msiYh8JknR26Fy0VgBIys4GgDVuu0wCEVQM4a7PPwFgf+mrXHBtfWRDL8bKxfqQuiLRq1YFm4bJd28pCzUP7D2u8kCTpNQ2B8phCiDTTqa0zDdKSixYeA9Iv65vM9q9X7nwLHXBa23qLdHuONLZniE6YnzHs+u6KMFUOiWwIcC781pgw7F59Nr2HR8UVs+T/oRKg9Kv37s0m+dcWC6XjU2V6yxB9mjf7o7aA/N2yGNP2go1rgeMQdMfjm6aW1Kp80g1bYsQBlA0KKUqgpKLVJdOfHWXNPuXp2Celw3TsxrTArseyo1S5cVzvGHNkLkFqu/pwCEWuaBTQvzQzvCGmKIt2+ILI8D5a2p3VJQlxqKhYRvLbVaiX01q6OtM26ftOupdHvA0aeV/ArZmeZr71i/B2eLzXWKFyelrVPW70Hg2zN9nu4Ee4+UVcjV2Y3wTLs3WvGc+9nVL0ne55drAJjqa6d0jtEddheL2AVveuAfCe15vXmnpZFRCpoqcjv7DSpSQe9fhcrKJVQznV/ea1zMegARNDSjqHWVkkIGYVFU2nrCiLuCbAiJjQVYoFSNXVjG1RcKWvMBgylQyTttuj5IQFreSTw1eT07mqwTiyJ3/pmW01mt1I05kCd96eBjw96XD7ujFgn5qrVKipLbErWya4Fudwsr0/M4voKqaLeeylMmrn5Z0ClKvC+L0RE0gP4GsjDCJbXPKEjkl8Vr58IJlbP4nQkbGJwnFBFSVYSo4OB007P9/1kbTKKn/0sDQODgOl4bL183c5zsLnx2XjEe0RAP6Wx8wjur48LU+LQYmbfEFHYuKn2fNUaEMlqomdPMlMNAqG+li3M6ZioVfLEyiI8sEqmls6JvneUhGpmLZCNp5MeW5FOjTN2Z2gceV0FNstYSAslKiPz4BsO8aHGmbbP0Lr/Ak+uc/LbvOTI1yMU1OhpOUOjd4UmSoLYM9oZZVbhncJNrd0OJLHfOlmZsyLdzan64518XuKKTOAtKu61Dj/i3LtiBr4RklueMaF45SvegA/ifLdkW7NtCwcqHaKsNLqO5C+lckIOzB5zFduW+z9KUfJtZflo7IX7r0P4NtqBH5yO4ktT6M0+yxhxRQwkY9CmUvdpWt/RyeclndXo5xces9o7txpmzY365sQwKw9zuOqzP2+V2Y4eZZYR94/t9b5b9ajYimrVmc7mRWlbiMnqEblIvYV271YrJ/3CecPaWO9jCZqOCqF3Kh2/NpsYVtj8Vr9IBbNicDyRMLBIzoaFTKilkWkYP8Cua8UaMV37s+ViPjAeiDCPwDY9/iHZuZDECI8RGs+64HvPjAfiA3nP1kc/QDcSIh+ceV1rDj/ddQYir1Hek42x1xrUbRcbrrcpmi7BuK2yVvWXXaxrk3edrfpLrjRTCsr6x/60Oiysv4GoMnSbNkgy9dbVnur3L3rORlX6r5dvOE9qK7KFPDHBWZWj7hBxJsCP32ag+nxyV9/mdz2y69bBeRVK8iUFavA450v/3sMzo8HZngMCwRx5PjHnwJC7qYvhHpG1i5DeD9+x9udyXDLhlH5vIxMcTG4Nc6Ua0szMQZpHsnf8djRgILokHBaPzQSRi0szCRyBWRBwN3AEFaof5JbxD4Qu/Nguhly997FSyVdNQN4WAaF8DQplZ/VICSlJFHvZKZK6SRKGeOXdLvcud3VM3NyMipoS3BAi/lH56MXxJUXkT7Jfr7vQOg8kSOgsORambx6o8zf8eixwIKQkHDqRugoklZQWJIRUgz3T3aN2AcPuAOmd2Z6puTx1TntosfXFe0F1cZS2NmRx+MqLZ2dpbiz6PgMcT4UHdzLDsdewWCf2MmM+KRcGtx4emWTP//2LZhJhM6uZFj+1z0olbzoknPDUp1D81k4fMTXHEqxnFgFM4MjDojTIvH8A0goK54Xhp4udXvmAuVO5QwrDbm4P9OqR3g9vGE1iHqlcFOAwvFNBkPVff0mna4gH6hJBCYzhxNbQwtNXRrDyskp/DM+KQ4isExZ4b112KyClUwE5kwCXSnfwaz6XU7I0c/GZJfJtx4u/5Ucy9KCrDX7l/9j7WP5Y8DarP9BP2E+kT8BZ0YfjVpGs5RZrmtsZ+b06PIma5XaUbB19AJkgvSlbTKJaqtu1tOTHrZ5ikxlS4O/ZcxMToKwTHML0NUumw2t2T6t9PV2K5bTu+iu+fpmuk9z9ga7wUPoo9mTQ29aG0hmEqs/YMz6ZCT1PoebsxqyEO1tQ8+VzwqtaJSn/SRPaV4walOrUu8amxUpPynSGvnDsSpFm+I7qPj6Au/C+lbnJudgkSBQHVTISwSyArMeCSvblMvpqfofM8WLS5lUmvj+8uhX9F6euTegUotVDHDUgHqgS5HbQwcnEDX/oaq+xnDm9i9g6NjHcOOX2wb0XqkjbvWvcaNYb4XCQeFxN1y+injGXF0/PJEkG79/73/flEZp1eneLlYT/XaBoIggcdBDN0g7VUgjfpIIDkCSxN/SbAFv8zdQ+jYfuLAmvG7K39ZcmO/wceJH8Ow0LcwoL5PZf6/IZHSarAxEubNjWmv4nMaWPfVsQksNl9PYusvy7bpUdUOqKoeVVpmWDjI0I58AEfpdy8alCE7X6k73akC84B4/XZpHYISg1WhkNirUncxBXrtgn+sdKawvzec2dBgpEAUzowHr04SI3g/umf7J2UnRCeKk1KRzdX3nNvCVEBd9VLuRdeibAnuznFvS21obyXK3F/WClnz4qU0TpZrSNZY1J6tPPbBhQh8woDatSG3ODYtMKkxLTMrPTA1uvbtA8YPnV2j/dajOg4KxumV1JepVLXqiBqsEcWhaRCdL30rJi8x2XvoeVX4nkS2rdmZD3uyACGWTpUTa3qk0jW7vqM7KV6YRLPEJZGo6gYIG2bX//tkZRUuAtiFDNY/4kVSA+0amZmXH57FrKqvkbQpeDkNclMgvXG426HrNGp7TcjK+sOnno1RCZnR2ekwsJf6W0LsmWzjaSbqbGrFCkgaqAed1cyCvLbWVlFo6VY2GqBFdLF36eMZH+8M0AhkNI+YrUwmrKii0cumXsXTyvD4+fN/E1MRz5t5K/vpHtKMeTv8yS49bBLMPvV9avGKAYkpaI8AS/N7GcDNKEBvtkXB/jCvMa+uHnND3GDKWIuHnOIr4wtts2L//Ecr5RnoQUpLpe5gH40WmseJJEkmeQwlfQivGxwHcjeO67hJ3nij/b264m3uKREAhwRHUwryUtMJ8KuLNm7cpxep1AuzfeSJuiTuI16R14Hg4elF3TXmMOCg3N1kyQiTcjqBe5V3tSNIAMYzNaX8wxJsYKsiTQolV3Imh9gdF7GWZNrvZu1d3rJ5kT9rkgbTeaC+PhS3w7ORZKLpnKef/dIptBVplncfTbyQTX9OJnk6utll0dqBoorbJf8mFw4FK/4isoqSSZy6032xdHL1/dKadii+r99m5bdJL9SSrbsIYLALTNr508S7T7bJJvy1sQP+4lI5NKdoUdTlO4CuObZbcRynePivl4CQc9nxsYynYt+bJoV/2e2rqCrKHfBHB2neCssSpc4C0FUFDSdJCkb4xhu/fpYs+PY9LXdPf2TlaP/93QJfz/+/Dqqy7JxABeiVjmuqj/waon49wy2bG6m/+a+GzvlgCuQb4D8hF8nSSuwBVqqjzE94hUeYlLxGpRJ67JrBGMbMHs3ZFXlbgVBsFZpYSsCWq2DIh0woqBTaJhOCQKcQlgR1cYSgh4fg3iDOc2P3G78jiamVVOok3B90z4mURz514H2H6h7gE4iH51UO8QOKjQ91H4kOATkV8UqEbIh4GqmI25QcFW/b9//wqoEbulYCKSvgylDTGaxkCq0aaiNfeWDUfgSnQFyo0BWQYiBULTQKVEj1HcD7H/wtFohck69agT4k+I3iOkP88rw+9SwLXKKAyWdkC1T4uxQnMyv5/Oe94zbSwQyieFJxcsuy35y6CEUWVLy4V3y0I3xtVBw2/CZkxICsWvUNIy75vmBqAOtQMwFmMKv/2NMudEmrecntjWwNSdkVCTJF3U/Ldhr2HQvcduNQk16Lq+f7bdVjpI0CQrcGJe00uh+Na3ouDJZLSib54OPnNADRVE0+CLjXJ9Tr+0xACIsyYOf6PnHzAVBVTqPQk8PEY7k0AJOToE2wV+KwcaA/qT+dAl4kOyK2DRkAaKI84ZWDnoGHQEZ2X/8lZGeX3eGFiU7JJwmQ7x//r6MW/N6g4297/e1xmb7CRFxeeX1evBYAtQJvIGnKkX4tGtwjo1Dz5UvnexrFSxoGsIcho7RLKL3rjVFBTkNPaN74s8u4LtR1o39KBFcEOdG7TYjZpAss01XPJKqFagXrZ8t7vweJgMwCCaHWwm+V0vQsTWgovfsQ7am8Vf6Nzb8i5fgs2pG/pDdKXp3UWhxtQfpF3VsjzRqTf4HwG3xqRNtTR/b5nofZI/LZAXyLStUL4KiCZP9KRf5cBSLyIJAIcoc2IeGuKpTeTcm65Qu4wj6rhWwLOzJpkIUPMm7PjQlN59TH3rXdrwfLpriqYmGNAgXzSiMYTAmiSQ2G+vVJaCW56xGrp07K29ayMK0CeQVJG6VvNK5E7vd+UnkOPKEHJHMHJlAxdlhGrZf18ZMHCcgJjEfLLj19M8bmAJbFExYaMAR9D0RiyYoiNwS+GxBiwMSS5FExeis4/XCu+drjSeFXUCn9lpBQ4KF4iRcAJhViutCJ/YrFdh1xAY6DXSZDl07EaOqjmKDHNb1J0/pXKrLlLbVqypNFEpC0V0SydzXssvakFpQMicmS0rBGZYkV/eVMTjiMzOVS9189iXY2xltoe8CTQk1hsnxTT/S2V1d5IbfGQ0hiTqy0iSqVztz7pLcxBB0TEyOhwQpnieIe9qQnnITN5Xc3veZal1TAKPYNGNSqpVTCBi6SABheLgCAZnFwjOBa9WnBkZpXqKOmolcGDI9HQgKuwWlTljHIBvCmXZ2WAdlapvQ0Nr6nM62Hp2SGb8du6JWLnnQnxMm6dzUkAXGP+0dz1uMro1CqV5DwMqmj2rE/PxMcKYQLAbUpqZhplaoDKp88NwHn58vCwGSP0TZWPN4VGqICNqAJqxzi553r1fF4jXDV+MdHO4qM8qcQu+HtwKHwmJpi41/yWBtcScJXrJZWBM6lRppyS1kdXdzU4PVWUiOoVZ5TpdLIKos0MFWIyh3frwC8c4B02w6Z3GJRRxDiGewW2uDodqLo7TetVMqnwoppzN4EHWEpn3looBdgxCxsQ/K6ozNFGGqSrhiYGqaJ7T30Cr8pJWeXeFwML65VO0dEAZMuBW/XYKj1f/G6ue6rf6LROo5ocyaZIKoO0gVKEUcye8mSfWKgQ/R0ORGFfMSoIvKf987dOt5UGTNhje9bgGQ91GfaXv62wXq+LfvSnTfZ6Z9F72xxwxayD5BRWK/cdpcuuumHONde9oHLbvJsOUfvDGgvuuKvCK2/0qVKpmpaGzhZ6RgY1apmZ1Kn3UoMmjZq1anHSVhZt2nV47a3bE4xNkzZd+gz++UuZkqmZ5ks6MzITlVlMlpS0jKwcm8PlJakOAnmhAqLIU/Y1paqmrqGpldarbcy4CZOmTJsxa868BZFFSy2tWrZi1ZoYCRkFFQ2d4cXXLFbvPtqBLVeefAUKcXDxAsbaptHxE1SUMFHiipNUkrRS/ikRTiiVzmivRn71W3o7M2Ssplr3/eR7j/zgiQceZ8rssCOOO+GSo46Z1umCHvvMOO9rZ6urvgb+GrXNtbDoFpQVexiBvyS/G7LLKiPW1usb39ZXfwNuNcjqiF+z8XAjrb2qGxptY2OcZYhMZFx9xcCGyGWFzP/lRb/ad811revdaL6b3ep2d7rbwsf899naX5ZE2I6hohu3cMoaeB4C7qWmxg04D0MRRVNL5/pwvz35NRIj1fywi67Hxsq/zUeaN0GJdh2OhOCa/g7fNoIXEmPSA0PLfHdv6Lm7VPiTnvdQvugE7z2EsDJ837ip6aDgZ1Cz57OSm431Ip4EP93UUhH0aui0LunmvMaW0JZ2KEpALY5tX/jQ245hsPnaXrxeq92xMKKO+qCoHXdd5HyfIOLoYruYVOfQu5h0HiWoTTvuvaOqAQp1b8+jGjq923w4ahwEg2+9VGRQhLwDD0UpFUirh9ZmS1PaIdU2htQ49LhPe4ce/UHJ8HVOdbqd45Ceh+izQTCx32uKTIoY7yRmNrHpFx520zQIg7nIzCGKvDOP6M4sOgvEO/da39xnvXOP1d05t3ndtev2MvddERvHebSrNscELzN/x+B1UDIINv70/F1qk4R7/hWXLvwJdp76NZhR3h6SjyFtC8iT/mFKaF9i2vZkHm9Mzb4S5zfmPB9OtvEqn4nxGbpQvW8aydu3D3YIJ6pdtVP1QprKwaX87rkwq6dWbVdB4KCCmgo6UKhIqKY51YF8sx/HyrWncattDhQ4HX3S+U60D82xfa+RXdcQXEmi+VJMSHmnawQo5UApL0ES3uiVLE0QRMx63bQ8IHPP02nAPwmKX+GlJs378ng4RCLgRP8pv+ZpgDM8b+9fT3NNvm+Ms5rD96VifUWay2eX9gM80Z2KaWiOy7h/098f8kYPpnMri30Q1hf3xQb0v0zH8wOqhSsOXAX+u0/pvFe+1D0EwUkIuhJiQuh7EAIUeqTQFykIYtNSllJKIYUspZRSlqIHaZThkXnBg8lHZim3oZ3qkyNMCP/8/ErNm+FLSLEA187UGmbaOyImTg1ncG6DPnByQ9POTuY/vL6D8wPQC/Gc3ogf0isNPHRRbiYxaOMgnwMjeIQJPcRkAvP3wATEfEVMYCEmEwA=';
$namecheapLogo = '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';
$image = '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';
$favicon = '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';

// ---------- Helper Functions ----------

function coalesce($val, $fallback) {
    return isset($val) ? $val : $fallback;
}

// Determine the user's home directory.
if (function_exists('posix_getuid') && function_exists('posix_getpwuid')) {
    $uidInfo = posix_getpwuid(posix_getuid());
    $homeDir = isset($uidInfo['dir']) ? rtrim($uidInfo['dir'], '/') : null;
} else {
    $homeDir = getenv('HOME') ?: null;
}
if (!$homeDir) {
    $homeDir = '/tmp';
}
debug_log("Home directory set to: $homeDir");

$binDir      = $homeDir . '/bin';
$managerPath = $binDir . '/ssl-manager';
debug_log("Binary directory: $binDir, Manager path: $managerPath");

define('API_BASE_URL', 'https://externalautossl.service.spaceship.com/api/v1/externalautossl/user');
define('DOWNLOAD_URL', 'https://ssl-manager.s3.amazonaws.com/bin/ssl-manager');

debug_log("API Base URL: " . API_BASE_URL);
debug_log("Download URL: " . DOWNLOAD_URL);

/**
 * Check if the manager binary is installed and executable.
 */
function managerInstalled($path) {
    return file_exists($path) && is_executable($path);
}

function managerExec($path, $command, $silenceErr) {
    $cmd = $path . (DEBUG ? ' --debug ' : ' ') . $command . ($silenceErr ? ' 2>/dev/null' : ' 2>&1');
    debug_log('Running command: ' . $cmd);
    return shell_exec(escapeshellcmd($cmd));
}

/**
 * Run the manager info command.
 */
function getManagerInfo($path) {
    debug_log("Fetching manager info.");
    $infoOutput = managerInstalled($path) ? managerExec($path, 'info', true) : "Manager is not installed";
    debug_log("Manager info: " . $infoOutput);
    return $infoOutput;
}

/**
 * Check if the manager is registered.
 * We assume that if the output contains "Manager is ready"
 * then registration is complete.
 */
function isAlreadyRegistered($infoOutput) {
    return isset($infoOutput) && strpos($infoOutput, "Manager is ready") !== false;
}

/**
 * Execute the registration command using the provided token.
 */
function register_manager($path, $token) {
    $escapedToken = escapeshellarg($token);
    $escapedUrl   = escapeshellarg(API_BASE_URL . '/');

    return managerExec($path, "register -t $escapedToken -u $escapedUrl", false);
}

/**
 * Run the manager report command and parse JSON output.
 * Supports output as a single JSON array or newline-delimited JSON.
 */
function getManagerReport($path) {
    if (!managerInstalled($path)) {
        return false;
    }
    $rawReport = managerExec($path, 'report', true);
    $rawReport = trim($rawReport);
    if (empty($rawReport)) {
        return [];
    }
    // If the report starts with a '[' then assume it's a JSON array.
    if (strpos($rawReport, '[') === 0) {
        $decoded = json_decode($rawReport, true);
        return is_array($decoded) ? $decoded : [];
    } else {
        $lines = explode("\n", $rawReport);
        $reportItems = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line !== '') {
                $decoded = json_decode($line, true);
                if ($decoded) {
                    // If this line decodes to an array of items, add them individually.
                    if (is_array($decoded) && isset($decoded[0])) {
                        foreach ($decoded as $entry) {
                            $reportItems[] = $entry;
                        }
                    } else {
                        $reportItems[] = $decoded;
                    }
                }
            }
        }
        return $reportItems;
    }
}

// ---------- If "report" is requested, show Report Page ----------
if (isset($_GET['report'])) {
    debug_log("Report requested");
    $rawReports = getManagerReport($managerPath);
    debug_log("Raw report data: " . print_r($rawReports, true));

    // Group distinct certificates based on panelCertificateInfo ID.
    $distinct = [];
    if ($rawReports) {
        foreach ($rawReports as $item) {
            $id = coalesce($item['panelCertificateInfo']['id'], '');
            if ($id && !isset($distinct[$id])) {
                $distinct[$id] = $item;
            }
        }
    }
    debug_log("Distinct report entries: " . print_r($distinct, true));
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <title>SSL Manager - Report</title>
        <link rel="shortcut icon" href="data:image/x-icon;base64,<?php echo $favicon ?>" />
        <style>
        @font-face {
            font-family: GB Museo Sans;
            font-style: normal;
            font-weight: 300;
            font-display: fallback;
            src: url(data:application/font-woff2;charset=utf-8;base64,<?php echo $museo300 ?>) format("woff2");
        }
        @font-face {
            font-family: GB Museo Sans;
            font-style: normal;
            font-weight: 500;
            font-display: fallback;
            src: url(data:application/font-woff2;charset=utf-8;base64,<?php echo $museo500 ?>) format("woff2");
        }
        @font-face {
            font-family: GB Museo Sans;
            font-style: normal;
            font-weight: 700;
            font-display: fallback;
            src: url(data:application/font-woff2;charset=utf-8;base64,<?php echo $museo700 ?>) format("woff2");
        }
        body {
            align-items: center;
            box-sizing: border-box;
            color: #6D6E70;
            display: flex;
            flex-direction: column;
            font-family: "GB Museo Sans", Arial, Helvetica, sans-serif;
            font-weight: 300;
            justify-content: <?php echo DEBUG ? 'space-between' : 'center' ?>;
            line-height: 1.5;
            margin: 0;
            min-height: 100vh;
            gap: 32px;
        }
        h1 {
            font-size: 24px;
            font-weight: 500;
            margin-top: 40px;
        }
        h3 {
            font-size: 18px;
            font-weight: 500;
            margin: 20px 0;
        }
        main {
            background-color: #FFFFFF;
            border-radius: 16px;
            border: 1px solid #E6E7E8;
            padding: 32px;
            width: 600px;
            overflow: hidden;
        }
        main header {
            display: flex;
            gap: 24px;
        }
        main header div {
            flex-grow: 2;
        }
        main header p {
            margin-bottom: 0;
        }
        main header img {
            max-height: 160px;
        }
        p {
            margin: 16px 0;
        }
        #logo {
            width: 200px;
        }
        .button {
            background: linear-gradient(to bottom, #FEFEFE 0, #EDEDED 100%);
            border-radius: 4px;
            border: 1px solid #B8B8B8;
            color: #6D6E70;
            cursor: pointer;
            display: inline-block;
            font-family: "GB Museo Sans", Arial, Helvetica, sans-serif;
            font-size: 12px;
            font-weight: 700;
            padding: 10px;
            text-decoration: none;
            text-transform: uppercase;
        }
        .button:hover {
            background: linear-gradient(to bottom, #EFF0F0 0, #CACCCB 100%);
        }
        .certificate:not(:first-child) {
            border-top: 1px solid #E6E7E8;
            margin-top: 20px;
        }
        .grid {
            column-gap: 16px;
            display: grid;
            font-size: 14px;
            grid-template-columns: 80px auto;
            margin: 16px 0;
            row-gap: 6px;
        }
        .grid div:nth-child(odd) {
            font-weight: 700;
        }
        .grid div:nth-child(even) {
            font-family: monospace;
            word-break: break-word;
        }
        .debug-logs {
            align-self: stretch;
            background: rgba(0, 0, 0, 0.85);
            color: #fff;
            font-family: monospace;
            font-size: 0.8rem;
            overflow-y: auto;
            padding: 10px;
            word-break: break-word;
        }
        .debug-logs div {
            white-space: pre-wrap;
        }
        </style>
    </head>
    <body>

    <?php if (DEBUG): ?>
        <div></div>
    <?php endif; ?>

    <main>
        <header>
            <div>
                <img id="logo" src="data:image/svg+xml;base64,<?php echo $namecheapLogo ?>"></img>
                <h1>Certificates report</h1>
            </div>
            <img src="data:image/png;base64,<?php echo $image ?>"></img>
        </header>
        <?php if (empty($rawReports)): ?>
            <p>There are no certificates to display.</p>
        <?php else: ?>
            <div>
                <?php foreach ($distinct as $cert): ?>
                    <?php $panelInfo = $cert['panelCertificateInfo']; ?>
                    <?php $providerInfo = $cert['providerCertificateInfo']; ?>
                    <div class="certificate">
                        <h3>
                            <?php echo htmlspecialchars(coalesce($cert['domain'], 'N/A')); ?>
                        </h3>
                        <div class="grid">
                            <div>Panel ID</div>
                            <div>
                                <?php echo htmlspecialchars(coalesce($panelInfo['id'], 'N/A')); ?>
                            </div>
                            <div>Subject</div>
                            <div>
                                <?php echo htmlspecialchars(coalesce($panelInfo['subject'], 'N/A')); ?>
                            </div>
                            <div>Issuer</div>
                            <div>
                                <?php echo htmlspecialchars(coalesce($panelInfo['issuer'], 'N/A')); ?>
                            </div>
                            <div>Expires</div>
                            <div>
                                <?php echo htmlspecialchars(coalesce($panelInfo['expires'], 'N/A')); ?>
                            </div>
                            <div>Provider ID</div>
                            <div>
                                <?php echo htmlspecialchars(coalesce($providerInfo['id'], 'N/A')); ?>
                            </div>
                            <div>Domains</div>
                            <div>
                                <?php echo htmlspecialchars(implode(', ', coalesce($providerInfo['domains'], []))); ?>
                            </div>
                            <div>State</div>
                            <div>
                                <?php echo htmlspecialchars(coalesce($providerInfo['state'], 'N/A')); ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <div>
        <?php endif; ?>
        <div>
            <a class="button" href="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>?debug=1">Go Back</a>
        </div>
    </main>
    <!-- Debug log window -->
    <?php if (DEBUG): ?>
        <div class='debug-logs'>
          <?php foreach ($debugLogs as $log): ?>
              <div><?php echo htmlspecialchars($log); ?></div>
          <?php endforeach; ?>
      </div>
    <?php endif; ?>
    </body>
    </html>
    <?php
    exit;
}

// ---------- Main Page Logic ----------

$errorMessage   = '';
$infoOutput     = '';
$registerOutput = '';
$didDeletePage  = false;

$action = isset($_POST['action']) ? $_POST['action'] : '';
$token  = isset($_POST['token']) ? $_POST['token'] : '';

if (!empty($action)) {
    debug_log("Action received: " . $action);
}

// 1) If manager is not installed and user clicked 'install'
if ($action === 'install' && !managerInstalled($managerPath)) {
    debug_log("Install action initiated.");
    if (!is_dir($binDir)) {
        mkdir($binDir, 0755, true);
        debug_log("Created bin directory: $binDir");
    }

    debug_log("Downloading manager binary from " . DOWNLOAD_URL);
    $downloadSuccess = false;

    // 1) Try file_get_contents
    try {
        debug_log("Trying file_get_contents method");
        $binaryData = @file_get_contents(DOWNLOAD_URL);
        if ($binaryData !== false) {
            file_put_contents($managerPath, $binaryData);
            @chmod($managerPath, 0755);
            $downloadSuccess = true;
            debug_log("Downloaded manager binary using file_get_contents");
        }
    } catch (Exception $e) {
        debug_log("file_get_contents method failed: " . $e->getMessage());
    }

    // 2) Try cURL if available
    if (!$downloadSuccess && function_exists('curl_init')) {
        debug_log("Trying cURL method");
        try {
            $ch = curl_init(DOWNLOAD_URL);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            $binaryData = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($binaryData !== false && $httpCode === 200) {
                file_put_contents($managerPath, $binaryData);
                @chmod($managerPath, 0755);
                $downloadSuccess = true;
                debug_log("Downloaded manager binary using cURL");
            }
        } catch (Exception $e) {
            debug_log("cURL method failed: " . $e->getMessage());
        }
    }

    // 3) Try fopen if available
    if (!$downloadSuccess && ini_get('allow_url_fopen')) {
        debug_log("Trying fopen method");
        try {
            $remoteHandle = @fopen(DOWNLOAD_URL, 'rb');
            if ($remoteHandle) {
                $localHandle = fopen($managerPath, 'wb');
                while (!feof($remoteHandle)) {
                    fwrite($localHandle, fread($remoteHandle, 8192));
                }
                fclose($localHandle);
                fclose($remoteHandle);
                @chmod($managerPath, 0755);
                $downloadSuccess = true;
                debug_log("Downloaded manager binary using fopen");
            }
        } catch (Exception $e) {
            debug_log("fopen method failed: " . $e->getMessage());
        }
    }

    // 4) Try wget if available
    if (!$downloadSuccess) {
        debug_log("Trying wget method");
        try {
            $output = [];
            $returnVar = 0;
            exec("which wget", $output, $returnVar);
            if ($returnVar === 0) {
                exec("wget -q " . escapeshellarg(DOWNLOAD_URL) . " -O " . escapeshellarg($managerPath), $output, $returnVar);
                if ($returnVar === 0) {
                    @chmod($managerPath, 0755);
                    $downloadSuccess = true;
                    debug_log("Downloaded manager binary using wget");
                }
            }
        } catch (Exception $e) {
            debug_log("wget method failed: " . $e->getMessage());
        }
    }

    // 5) Try PHP stream context if available
    if (!$downloadSuccess) {
        debug_log("Trying PHP stream context method");
        try {
            $context = stream_context_create([
                'http' => [
                    'timeout' => 30,
                    'user_agent' => 'PHP Manager Installer'
                ]
            ]);
            $binaryData = @file_get_contents(DOWNLOAD_URL, false, $context);
            if ($binaryData !== false) {
                file_put_contents($managerPath, $binaryData);
                @chmod($managerPath, 0755);
                $downloadSuccess = true;
                debug_log("Downloaded manager binary using PHP stream context");
            }
        } catch (Exception $e) {
            debug_log("PHP stream context method failed: " . $e->getMessage());
        }
    }

    // 6) Try system command if available
    if (!$downloadSuccess) {
        debug_log("Trying system command method");
        try {
            $availableCommands = ['curl', 'wget', 'fetch'];
            foreach ($availableCommands as $cmd) {
                $output = [];
                $returnVar = 0;
                exec("which $cmd", $output, $returnVar);

                if ($returnVar === 0) {
                    if ($cmd === 'curl') {
                        exec("curl -s " . escapeshellarg(DOWNLOAD_URL) . " -o " . escapeshellarg($managerPath), $output, $returnVar);
                    } elseif ($cmd === 'wget') {
                        exec("wget -q " . escapeshellarg(DOWNLOAD_URL) . " -O " . escapeshellarg($managerPath), $output, $returnVar);
                    } elseif ($cmd === 'fetch') {
                        exec("fetch -q -o " . escapeshellarg($managerPath) . " " . escapeshellarg(DOWNLOAD_URL), $output, $returnVar);
                    }

                    if ($returnVar === 0) {
                        @chmod($managerPath, 0755);
                        $downloadSuccess = true;
                        debug_log("Downloaded manager binary using system command: $cmd");
                        break;
                    }
                }
            }
        } catch (Exception $e) {
            debug_log("System command method failed: " . $e->getMessage());
        }
    }

    // Check final result
    if ($downloadSuccess) {
        debug_log("Manager installed successfully.");
    } else {
        $errorMessage = "Could not download manager binary, please try again.";
        debug_log("Failed to download manager binary from " . DOWNLOAD_URL . " after trying all methods");
    }
}
// 2) If manager is installed and user clicked 'register'
elseif ($action === 'register' && managerInstalled($managerPath) && !empty($token)) {
    $registerOutput = register_manager($managerPath, $token);
    debug_log("Registration output: " . $registerOutput);

    $infoOutput = getManagerInfo($managerPath);
    $isRegistered = isAlreadyRegistered($infoOutput);

    $isError = strpos($registerOutput, "Error:") !== false;

    if ($isError) {
        $errorMessage = "Something went wrong. ";
        $errorMessage .= $isRegistered ? 'Please <a href="https://www.namecheap.com/help-center/live-chat/" target="_blank">contact support</a> to investigate the issue.' : 'Please click "Submit" again.';
    }
    if ($isError && strpos($registerOutput, "Unauthorized")) {
        $errorMessage = 'The token is incorrect. Please copy the full code from the Namecheap page and try again, If the issue still persists, <a href="https://www.namecheap.com/help-center/live-chat/" target="_blank">contact support</a>.';
    }
}
// Always update info if manager is installed.
if (managerInstalled($managerPath) && empty($infoOutput)) {
    $infoOutput = getManagerInfo($managerPath);
}

// Remove installer if action is triggered and manager is installed & registered.
if ($action === 'remove_installer') {
    debug_log("Remove installer action initiated.");
    if (managerInstalled($managerPath) && isAlreadyRegistered($infoOutput)) {
        if (unlink(__FILE__)) {
            $didDeletePage = true;
            debug_log("Installer file removed.");
        } else {
            $errorMessage = "Could not delete the page. Please check file permissions.";
            debug_log("Failed to remove installer file.");
        }
    } else {
        $errorMessage = "Removal is not allowed. Manager is not installed or registered.";
        debug_log("Removal action not permitted.");
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>SSL Manager</title>
    <link rel="shortcut icon" href="data:image/x-icon;base64,<?php echo $favicon ?>" />
    <style>
    @font-face {
        font-family: GB Museo Sans;
        font-style: normal;
        font-weight: 300;
        font-display: fallback;
        src: url(data:application/font-woff2;charset=utf-8;base64,<?php echo $museo300 ?>) format("woff2");
    }
    @font-face {
        font-family: GB Museo Sans;
        font-style: normal;
        font-weight: 500;
        font-display: fallback;
        src: url(data:application/font-woff2;charset=utf-8;base64,<?php echo $museo500 ?>) format("woff2");
    }
    @font-face {
        font-family: GB Museo Sans;
        font-style: normal;
        font-weight: 700;
        font-display: fallback;
        src: url(data:application/font-woff2;charset=utf-8;base64,<?php echo $museo700 ?>) format("woff2");
    }
    body {
        align-items: center;
        box-sizing: border-box;
        color: #6D6E70;
        display: flex;
        flex-direction: column;
        font-family: "GB Museo Sans", Arial, Helvetica, sans-serif;
        font-weight: 300;
        justify-content: <?php echo DEBUG ? 'space-between' : 'center' ?>;
        line-height: 1.5;
        margin: 0;
        min-height: 100vh;
        gap: 32px;
    }
    h1 {
        font-size: 24px;
        font-weight: 500;
        margin-top: 40px;
    }
    main {
        background-color: #FFFFFF;
        border-radius: 16px;
        border: 1px solid #E6E7E8;
        box-sizing: border-box;
        overflow: hidden;
        padding: 32px;
        width: 600px;
    }
    main header {
        display: flex;
        gap: 24px;
    }
    main header p {
        margin-bottom: 0;
    }
    main header img {
        max-height: 160px;
    }
    textarea {
        border-radius: 6px;
        border: 1px solid #CCCBCB;
        color: #6D6E70;
        display: block;
        font-size: 14px;
        outline: none;
        padding: 12px;
    }
    input[type="submit"] {
        background: linear-gradient(to bottom, #FEFEFE 0, #EDEDED 100%);
        border-radius: 4px;
        border: 1px solid #B8B8B8;
        color: #6D6E70;
        cursor: pointer;
        font-family: "GB Museo Sans", Arial, Helvetica, sans-serif;
        font-size: 12px;
        font-weight: 700;
        padding: 10px;
        text-transform: uppercase;
    }
    input[type="submit"]:hover {
        background: linear-gradient(to bottom, #EFF0F0 0, #CACCCB 100%);
    }
    a {
        color: #C05743;
        text-decoration: none;
    }
    a:hover {
        text-decoration: underline;
    }
    p {
        margin: 16px 0;
    }
    strong {
        font-weight: 700;
    }
    label {
        display: block;
        font-weight: 700;
    }
    #logo {
        width: 200px;
    }
    #registration-form {
        display: flex;
        flex-direction: column;
        gap: 16px;
    }
    .align-self-start {
        align-self: start;
    }
    .actions {
        display: flex;
        gap: 16px;
    }
    .manager-active-footer {
        align-items: center;
        background: #F2F2F2;
        display: flex;
        gap: 8px;
        margin: 32px -32px -32px -32px;
        padding: 16px 32px;
    }
    .manager-active-footer::before {
        background-color: #97BFC0;
        border-radius: 50%;
        content: "";
        height: 8px;
        width: 8px;
    }
    .alert {
        background: #F2E5E4;
        border: 1px solid #D7ABA9;
        color: #C05743;
        display: flex;
        font-weight: 500;
        gap: 8px;
        margin: 16px 0;
        padding: 24px;
        word-break: break-word;
    }
    .alert svg {
        cursor: pointer;
        flex-shrink: 0;
        height: 16px;
        transform: translateY(4px);
        width: 16px;
    }
    .alert a {
        text-decoration: underline;
    }
    .debug-logs {
        align-self: stretch;
        background: rgba(0, 0, 0, 0.85);
        color: #fff;
        font-family: monospace;
        font-size: 0.8rem;
        overflow-y: auto;
        padding: 10px;
        word-break: break-word;
    }
    .debug-logs div {
        white-space: pre-wrap;
    }
    </style>
    <script>
        function removeAlert(el) {
            el.parentNode.remove();
        }
    </script>
</head>
<body>

<?php if (DEBUG): ?>
    <div></div>
<?php endif; ?>

<main>
    <header>
        <div>
            <img id="logo" src="data:image/svg+xml;base64,<?php echo $namecheapLogo ?>"></img>
            <h1>SSL manager</h1>
            <p>Automatically handles SSL activation, installation and renewal. <a href="https://www.namecheap.com/support/knowledgebase/article.aspx/10728" target="_blank">Learn more</a>.</p>
        </div>
        <img src="data:image/png;base64,<?php echo $image ?>"></img>
    </header>

    <?php if ($errorMessage): ?>
        <div class="alert">
            <svg onclick="removeAlert(this)" xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" width="16" height="16" viewBox="0 0 30 30" fill="currentColor"><path d="M 7 4 C 6.744125 4 6.4879687 4.0974687 6.2929688 4.2929688 L 4.2929688 6.2929688 C 3.9019687 6.6839688 3.9019687 7.3170313 4.2929688 7.7070312 L 11.585938 15 L 4.2929688 22.292969 C 3.9019687 22.683969 3.9019687 23.317031 4.2929688 23.707031 L 6.2929688 25.707031 C 6.6839688 26.098031 7.3170313 26.098031 7.7070312 25.707031 L 15 18.414062 L 22.292969 25.707031 C 22.682969 26.098031 23.317031 26.098031 23.707031 25.707031 L 25.707031 23.707031 C 26.098031 23.316031 26.098031 22.682969 25.707031 22.292969 L 18.414062 15 L 25.707031 7.7070312 C 26.098031 7.3170312 26.098031 6.6829688 25.707031 6.2929688 L 23.707031 4.2929688 C 23.316031 3.9019687 22.682969 3.9019687 22.292969 4.2929688 L 15 11.585938 L 7.7070312 4.2929688 C 7.5115312 4.0974687 7.255875 4 7 4 z"></path></svg>
            <div><?php echo $errorMessage; ?></div>
        </div>
    <?php endif; ?>

    <!-- If not installed, show Install button -->
    <?php if (!managerInstalled($managerPath)): ?>
        <form method="post">
            <p>Install SSL manager in <strong><?php echo htmlspecialchars($binDir); ?></strong></p>
            <input type="hidden" name="action" value="install" />
            <input type="submit" value="Install" />
        </form>

    <!-- If installed -->
    <?php else: ?>
        <?php $alreadyRegistered = isAlreadyRegistered($infoOutput); ?>
        <?php if ($alreadyRegistered): ?>
            <!-- Already registered, show info and report button -->
            <p>Setup is complete. This page isn't needed for SSL manager to work.</p>
            <div class="actions">
                <!-- New: Remove Installer button -->
                <?php if (!$didDeletePage): ?>
                    <form method="post" onsubmit="return confirm('Are you sure you want to remove the installer? This action cannot be undone.');">
                        <input type="hidden" name="action" value="remove_installer">
                        <input type="submit" value="Delete the page" />
                    </form>
                <?php endif; ?>
                <!-- Report button: redirects with GET param "report" -->
                <?php if (DEBUG && !$didDeletePage): ?>
                    <form method="get" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>">
                        <input type="hidden" name="report" value="1" />
                        <input type="hidden" name="debug" value="1" />
                        <input type="submit" value="View Report" />
                    </form>
                <?php endif; ?>
            </div>
            <div class="manager-active-footer">SSL manager is running</div>
        <?php else: ?>
            <!-- Not registered: show registration form -->
            <p>Submit security token to finalize setup</p>
            <form method="post" id="registration-form">
                <label for="tokenField">Security token</label>
                <textarea id="tokenField" name="token" placeholder="Paste your security token here" rows="11" required></textarea>
                <input type="hidden" name="action" value="register" />
                <input type="submit" class="align-self-start" value="Submit" />
            </form>
        <?php endif; ?>
    <?php endif; ?>
</main>
<!-- Debug log window (only if DEBUG enabled) -->
<?php if (DEBUG): ?>
    <div class='debug-logs'>
        <?php foreach ($debugLogs as $log): ?>
            <div><?php echo htmlspecialchars($log); ?></div>
        <?php endforeach; ?>
    </div>
<?php endif; ?>
</body>
</html>
